<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Faker\Factory as Faker;
use Illuminate\Support\Str;

class InsurancePoliciesSeeder extends Seeder
{
    public function run()
    {
        // Force English locale to ensure all text is in English
        $faker = Faker::create('en_US');

        // Ethiopian insurance policy types
        $policyTypes = ['Life', 'Motor', 'Health', 'Property', 'Travel', 'Microinsurance'];
        $coverageTypes = ['individual', 'family', 'group'];

        // Payment frequencies
        $premiumFrequencies = ['monthly', 'quarterly', 'semi-annually', 'annually'];

        // Payment methods common in Ethiopia (ETB currency implied)
        $paymentMethods = ['bank_transfer', 'mobile_money', 'cash', 'credit_card'];

        // Policy statuses
        $statuses = ['active', 'inactive', 'cancelled', 'expired', 'pending'];

        // Policy sources in Ethiopian insurance context
        $sources = ['agent', 'branch', 'online', 'community_outreach'];

        for ($i = 1; $i <= 50; $i++) {
            $startDate = $faker->dateTimeBetween('-3 years', 'now');
            $endDate = (clone $startDate)->modify('+1 year');
            $issueDate = (clone $startDate)->modify('-10 days');
            $maturityDate = (clone $startDate)->modify('+10 years');
            $renewalDate = (clone $endDate)->modify('+1 day');

            // Ethiopian-style policy number prefix + unique number
            $policyPrefix = $faker->randomElement(['ETH', 'NISCO', 'ADDIS', 'AXA', 'DELTA']);
            $policyNumber = $policyPrefix . '-' . strtoupper(Str::random(3)) . $faker->unique()->numerify('###');

            DB::table('policies')->insert([
                'customer_id' => $faker->numberBetween(1, 100),
                'corporate_customer_id' => $faker->optional()->numberBetween(1, 20),
                'policy_number' => $policyNumber,
                'policy_type' => $faker->randomElement($policyTypes),
                'coverage_type' => $faker->randomElement($coverageTypes),

                // Use simple English sentence instead of catchPhrase
                'policy_name' => ucfirst($faker->words(3, true)), // e.g. "Secure Life Plan"

                // Simple English paragraph
                'description' => $faker->sentence(12), 

                'start_date' => $startDate->format('Y-m-d'),
                'end_date' => $endDate->format('Y-m-d'),
                'issue_date' => $issueDate->format('Y-m-d'),
                'maturity_date' => $maturityDate->format('Y-m-d'),
                'renewal_date' => $renewalDate->format('Y-m-d'),

                // Amounts are in Ethiopian Birr (ETB)
                'sum_assured' => $faker->randomFloat(2, 100000, 10000000),  // ETB 100,000 to 10,000,000
                'premium_amount' => $faker->randomFloat(2, 500, 100000),    // ETB 500 to 100,000

                'premium_frequency' => $faker->randomElement($premiumFrequencies),
                'payment_method' => $faker->randomElement($paymentMethods),
                'auto_renew' => $faker->boolean(60),
                'status' => $faker->randomElement($statuses),

                // Optional cancellation reason in English
                'cancellation_reason' => $faker->optional()->sentence(6),

                'number_of_beneficiaries' => $faker->numberBetween(0, 5),

                // Attachments as JSON array of image URLs (generic, English)
                'attachments' => json_encode([
                    $faker->imageUrl(640, 480, 'insurance', true),
                    $faker->imageUrl(640, 480, 'documents', true)
                ]),

                // Optional English notes
                'notes' => $faker->optional()->sentence(10),

                'source' => $faker->randomElement($sources),

                'created_by' => $faker->numberBetween(1, 5),
                'updated_by' => $faker->numberBetween(1, 5),

                'created_at' => now(),
                'updated_at' => now(),
                'deleted_at' => null,
            ]);
        }
    }
}
